<?php

defined('ABSPATH') || exit;

use Atlets_Theme_Helper as Atlets;

/**
 * Atlets Dynamic Styles
 *
 *
 * @package atlets\core\class
 * @author WebGeniusLab <webgeniuslab@gmail.com>
 * @since 1.0.0
 * @version 1.0.2
 */
class Atlets_Dynamic_Styles
{
    protected static $instance;

    private $theme_slug;
    private $template_directory_uri;
    private $use_minified;
    private $enqueued_stylesheets = [];
    private $header_page_id;
    private ?string $gradient_enabled;

    public function __construct()
    {
        $this->theme_slug = $this->get_theme_slug();
        $this->template_directory_uri = get_template_directory_uri();
        $this->use_minified = Atlets::get_option('use_minified') ? '.min' : '';
        $this->header_type = Atlets::get_option('header_type');
        $this->gradient_enabled = Atlets::get_mb_option('use-gradient', 'mb_page_colors_switch', 'custom');

        $this->enqueue_styles_and_scripts();
        $this->add_body_classes();
    }

    public function get_theme_slug()
    {
        return str_replace('-child', '', wp_get_theme()->get('TextDomain'));
    }

    public function enqueue_styles_and_scripts()
    {
        //* Elementor Compatibility
        add_action('wp_enqueue_scripts', [$this, 'get_elementor_css_theme_builder']);
        add_action('wp_enqueue_scripts', [$this, 'elementor_column_fix']);

        add_action('wp_enqueue_scripts', [$this, 'frontend_stylesheets']);
        add_action('wp_enqueue_scripts', [$this, 'frontend_scripts']);

        add_action('admin_enqueue_scripts', [$this, 'admin_stylesheets']);
        add_action('admin_enqueue_scripts', [$this, 'admin_scripts']);
    }

    public function get_elementor_css_theme_builder()
    {
        $current_post_id = get_the_ID();
        $css_files = [];

        $locations[] = $this->get_elementor_css_cache_header();
        $locations[] = $this->get_elementor_css_cache_header_sticky();
        $locations[] = $this->get_elementor_css_cache_footer();
        $locations[] = $this->get_elementor_css_cache_side_panel();

        foreach ($locations as $location) {
            //* Don't enqueue current post here (let the preview/frontend components to handle it)
            if ($location && $current_post_id !== $location) {
                $css_file = new \Elementor\Core\Files\CSS\Post($location);
                $css_files[] = $css_file;
            }
        }

        if (!empty($css_files)) {
            \Elementor\Plugin::$instance->frontend->enqueue_styles();
            foreach ($css_files as $css_file) {
                $css_file->enqueue();
            }
        }
    }

    public function get_elementor_css_cache_header()
    {
        if (
            !apply_filters('atlets/header/enable', true)
            || !class_exists('\Elementor\Core\Files\CSS\Post')
        ) {
            // Bailtout.
            return;
        }

        if (
            $this->RWMB_is_active()
            && 'custom' === rwmb_meta('mb_customize_header_layout')
            && 'default' !== rwmb_meta('mb_header_content_type')
        ) {
            $this->header_type = 'custom';
            $this->header_page_id = rwmb_meta('mb_customize_header');
        } else {
            $this->header_page_id = Atlets::get_option('header_page_select');
        }

        if ('custom' === $this->header_type) {
            return $this->multi_language_support($this->header_page_id, 'header');
        }
    }

    /**
     * @version 1.0.2
     */
    public function get_elementor_css_cache_header_sticky()
    {
        if (
            ! apply_filters( 'atlets/header/enable', true )
            || 'custom' !== $this->header_type
            || ! class_exists( '\Elementor\Core\Files\CSS\Post' )
        ) {
            // Bailtout.
            return;
        }

        $header_sticky_page_id = '';

        if (
            $this->RWMB_is_active()
            && 'custom' === rwmb_meta( 'mb_customize_header_layout' )
            && 'default' !== rwmb_meta( 'mb_sticky_header_content_type' )
        ) {
            $header_sticky_page_id = rwmb_meta( 'mb_customize_sticky_header' );
        } elseif ( Atlets::get_option( 'header_sticky' ) ) {
            $header_sticky_page_id = Atlets::get_option( 'header_sticky_page_select' );
        }

        return $this->multi_language_support( $header_sticky_page_id, 'header' );
    }

    public function get_elementor_css_cache_footer()
    {
        $footer = apply_filters('atlets/footer/enable', true);
        $footer_switch = $footer['footer_switch'] ?? '';

        if (
            !$footer_switch
            || 'pages' !== Atlets::get_mb_option('footer_content_type', 'mb_footer_switch', 'on')
            || !class_exists('\Elementor\Core\Files\CSS\Post')
        ) {
            // Bailout.
            return;
        }

        $footer_page_id = Atlets::get_mb_option('footer_page_select', 'mb_footer_switch', 'on');

        return $this->multi_language_support($footer_page_id, 'footer');
    }

    public function get_elementor_css_cache_side_panel()
    {
        if (
            !Atlets::get_option('side_panel_enable')
            || 'pages' !== Atlets::get_mb_option('side_panel_content_type', 'mb_customize_side_panel', 'custom')
            || !class_exists('\Elementor\Core\Files\CSS\Post')
        ) {
            // Bailout.
            return;
        }

        $sp_page_id = Atlets::get_mb_option('side_panel_page_select', 'mb_customize_side_panel', 'custom');

        return $this->multi_language_support($sp_page_id, 'side_panel');
    }

    public function multi_language_support($page_id, $page_type)
    {
        if (!$page_id) {
            // Bailout.
            return;
        }

        $page_id = intval($page_id);

        if (class_exists('Polylang') && function_exists('pll_current_language')) {
            $currentLanguage = pll_current_language();
            $translations = PLL()->model->post->get_translations($page_id);

            $polylang_id = $translations[$currentLanguage] ?? '';
            $page_id = $polylang_id ?: $page_id;
        }

        if (class_exists('SitePress')) {
            $page_id = wpml_object_id_filter($page_id, $page_type, false, ICL_LANGUAGE_CODE);
        }

        return $page_id;
    }

    public function elementor_column_fix()
    {
        $css = '.elementor-container > .elementor-row > .elementor-column > .elementor-element-populated,'
            . '.elementor-container > .elementor-column > .elementor-element-populated {'
                . 'padding-top: 0;'
                . 'padding-bottom: 0;'
            . '}';

        $css .= '.elementor-column-gap-default > .elementor-row > .elementor-column > .elementor-element-populated,'
            . '.elementor-column-gap-default > .elementor-column > .elementor-element-populated {'
                . 'padding-left: 15px;'
                . 'padding-right: 15px;'
            . '}';

        wp_add_inline_style('elementor-frontend', $css);
    }

    public function frontend_stylesheets()
    {
        wp_enqueue_style($this->theme_slug . '-theme-info', get_bloginfo('stylesheet_url'));

        $this->enqueue_css_variables();
        $this->enqueue_additional_styles();
        $this->enqueue_style('main', '/css/');
        $this->enqueue_pluggable_styles();
        $this->enqueue_style('responsive', '/css/', $this->enqueued_stylesheets);
        $this->enqueue_style('dynamic', '/css/', $this->enqueued_stylesheets);
    }

    public function enqueue_css_variables()
    {
        return wp_add_inline_style(
            $this->theme_slug . '-theme-info',
            $this->retrieve_css_variables_and_extra_styles()
        );
    }

    public function enqueue_additional_styles()
    {
        wp_enqueue_style('font-awesome-5-all', $this->template_directory_uri . '/css/font-awesome-5.min.css');
        wp_enqueue_style('atlets-flaticon', $this->template_directory_uri . '/fonts/flaticon/flaticon.css');
    }

    public function retrieve_css_variables_and_extra_styles()
    {
        $root_vars = $extra_css = '';

        /**
         * Color Variables
         */
        if (
            class_exists('RWMB_Loader')
            && 'custom' == Atlets::get_mb_option('page_colors_switch', 'mb_page_colors_switch', 'custom')
        ) {
            $theme_primary_color = Atlets::get_mb_option('page_theme_color');
            $theme_secondary_color = Atlets::get_mb_option('page_theme_secondary_color');
            $theme_tertiary_color = Atlets::get_mb_option('page_theme_tertiary_color');

	        $button_color_idle = Atlets::get_mb_option('page_button_idle_color');
            $button_color_hover = Atlets::get_mb_option('page_button_hover_color');

            $bg_body = Atlets::get_mb_option('body_background_color');

            $scroll_up_arrow_color = Atlets::get_mb_option('scroll_up_arrow_color');
            $scroll_up_bg_color = Atlets::get_mb_option('scroll_up_bg_color');

            $this->gradient_enabled && $theme_gradient_from = Atlets::get_mb_option('theme-gradient-from');
            $this->gradient_enabled && $theme_gradient_to = Atlets::get_mb_option('theme-gradient-to');
        } else {
            $theme_primary_color = esc_attr(Atlets::get_option('theme-primary-color'));
            $theme_secondary_color = esc_attr(Atlets::get_option('theme-secondary-color'));
            $theme_tertiary_color = esc_attr(Atlets::get_option('theme-tertiary-color'));

	        $button_color_idle = esc_attr(Atlets::get_option('button-idle-color'));
            $button_color_hover = esc_attr(Atlets::get_option('button-hover-color'));

            $scroll_up_arrow_color = Atlets::get_option('scroll_up_arrow_color');
            $scroll_up_bg_color = Atlets::get_option('scroll_up_bg_color');

            $bg_body = esc_attr(Atlets::get_option('body-background-color'));

            $this->gradient_enabled && $theme_gradient = Atlets::get_option('theme-gradient');
            $this->gradient_enabled && $second_gradient = Atlets::get_option('second-gradient');
            $theme_gradient_from = $theme_gradient['from'] ?? '';
            $theme_gradient_to = $theme_gradient['to'] ?? '';
        }

        $root_vars .= '--atlets-primary-color: ' . esc_attr($theme_primary_color) . ';';
        $root_vars .= '--atlets-secondary-color: ' . esc_attr($theme_secondary_color) . ';';
        $root_vars .= '--atlets-tertiary-color: ' . esc_attr($theme_tertiary_color) . ';';

        $root_vars .= '--atlets-button-color-idle: ' . esc_attr($button_color_idle) . ';';
        $root_vars .= '--atlets-button-color-hover: ' . esc_attr($button_color_hover) . ';';

        $root_vars .= '--atlets-back-to-top-color: ' . esc_attr($scroll_up_arrow_color) . ';';
        $root_vars .= '--atlets-back-to-top-background: ' . esc_attr($scroll_up_bg_color) . ';';

        $root_vars .= '--atlets-body-background: ' . esc_attr($bg_body) . ';';

        $hsl_primary_color = Atlets::hexToHsl($theme_primary_color);
        $root_vars .= '--atlets-primary-color-lighter: ' . esc_attr(Atlets::shaded($hsl_primary_color, -5)) . ';';
        $root_vars .= '--atlets-primary-color-shaded: ' . esc_attr(Atlets::shaded($hsl_primary_color, 10)) . ';';

        $root_vars .= '--atlets-secondary-rgb: ' . esc_attr(Atlets::HexToRGB($theme_secondary_color)) . ';';

        $hsl_secondary_color = Atlets::hexToHsl($theme_secondary_color);
        $root_vars .= '--atlets-secondary-color-shaded: ' . esc_attr(Atlets::shaded($hsl_secondary_color, 10)) . ';';
        //* ↑ color variables

        /**
         * Headings Variables
         */
        $header_font = Atlets::get_option('header-font');
        $root_vars .= '--atlets-header-font-family: ' . (esc_attr($header_font['font-family'] ?? '')) . ';';
        $root_vars .= '--atlets-header-font-weight: ' . (esc_attr($header_font['font-weight'] ?? '')) . ';';
        $root_vars .= '--atlets-header-font-color: ' . (esc_attr($header_font['color'] ?? '')) . ';';

        for ($i = 1; $i <= 6; $i++) {
            ${'header-h' . $i} = Atlets::get_option('header-h' . $i);

            $root_vars .= '--atlets-h' . $i . '-font-family: ' . (esc_attr(${'header-h' . $i}['font-family'] ?? '')) . ';';
            $root_vars .= '--atlets-h' . $i . '-font-size: ' . (esc_attr(${'header-h' . $i}['font-size'] ?? '')) . ';';
            $root_vars .= '--atlets-h' . $i . '-line-height: ' . (esc_attr(${'header-h' . $i}['line-height'] ?? '')) . ';';
            $root_vars .= '--atlets-h' . $i . '-font-weight: ' . (esc_attr(${'header-h' . $i}['font-weight'] ?? '')) . ';';
            $root_vars .= '--atlets-h' . $i . '-text-transform: ' . (esc_attr(${'header-h' . $i}['text-transform'] ?? '')) . ';';
        }
        //* ↑ headings variables

        /**
         * Content Variables
         */
        $main_font = Atlets::get_option('main-font');
        $content_font_size = $main_font['font-size'] ?? '';
        $content_line_height = $main_font['line-height'] ?? '';
        $content_line_height = $content_line_height ? round(((int) $content_line_height / (int) $content_font_size), 3) : '';

        $root_vars .= '--atlets-content-font-family: ' . (esc_attr($main_font['font-family'] ?? '')) . ';';
        $root_vars .= '--atlets-content-font-size: ' . esc_attr($content_font_size) . ';';
        $root_vars .= '--atlets-content-line-height: ' . esc_attr($content_line_height) . ';';
        $root_vars .= '--atlets-content-font-weight: ' . (esc_attr($main_font['font-weight'] ?? '')) . ';';
        $root_vars .= '--atlets-content-color: ' . (esc_attr($main_font['color'] ?? '')) . ';';
        //* ↑ content variables

        /**
         * Menu Variables
         */
        $menu_font = Atlets::get_option('menu-font');
        $root_vars .= '--atlets-menu-font-family: ' . (esc_attr($menu_font['font-family'] ?? '')) . ';';
        $root_vars .= '--atlets-menu-font-size: ' . (esc_attr($menu_font['font-size'] ?? '')) . ';';
        $root_vars .= '--atlets-menu-line-height: ' . (esc_attr($menu_font['line-height'] ?? '')) . ';';
        $root_vars .= '--atlets-menu-font-weight: ' . (esc_attr($menu_font['font-weight'] ?? '')) . ';';
        //* ↑ menu variables

        /**
         * Submenu Variables
         */
        $sub_menu_font = Atlets::get_option('sub-menu-font');
        $root_vars .= '--atlets-submenu-font-family: ' . (esc_attr($sub_menu_font['font-family'] ?? '')) . ';';
        $root_vars .= '--atlets-submenu-font-size: ' . (esc_attr($sub_menu_font['font-size'] ?? '')) . ';';
        $root_vars .= '--atlets-submenu-line-height: ' . (esc_attr($sub_menu_font['line-height'] ?? '')) . ';';
        $root_vars .= '--atlets-submenu-font-weight: ' . (esc_attr($sub_menu_font['font-weight'] ?? '')) . ';';
        $root_vars .= '--atlets-submenu-color: ' . (esc_attr(Atlets::get_option('sub_menu_color') ?? '')) . ';';
        $root_vars .= '--atlets-submenu-background: ' . (esc_attr(Atlets::get_option('sub_menu_background')['rgba'] ?? '')) . ';';

        $root_vars .= '--atlets-submenu-mobile-color: ' . (esc_attr(Atlets::get_option('mobile_sub_menu_color') ?? '')) . ';';
        $root_vars .= '--atlets-submenu-mobile-background: ' . (esc_attr(Atlets::get_option('mobile_sub_menu_background')['rgba'] ?? '')) . ';';
        $root_vars .= '--atlets-submenu-mobile-overlay: ' . (esc_attr(Atlets::get_option('mobile_sub_menu_overlay')['rgba'] ?? '')) . ';';

        $sub_menu_border = Atlets::get_option('header_sub_menu_bottom_border');
        if ($sub_menu_border) {
            $sub_menu_border_height = Atlets::get_option('header_sub_menu_border_height')['height'] ?? '';
            $sub_menu_border_color = Atlets::get_option('header_sub_menu_bottom_border_color')['rgba'] ?? '';

            $extra_css .= '.primary-nav ul li ul li:not(:last-child),'
                . '.sitepress_container > .wpml-ls ul ul li:not(:last-child) {'
                . ($sub_menu_border_height ? 'border-bottom-width: ' . (int) esc_attr($sub_menu_border_height) . 'px;' : '')
                . ($sub_menu_border_color ? 'border-bottom-color: ' . esc_attr($sub_menu_border_color) . ';' : '')
                    . 'border-bottom-style: solid;'
                . '}';
        }
        //* ↑ submenu variables

        /**
         * Additional Font Variables
         */
        $extra_font = Atlets::get_option('additional-font');
        empty($extra_font['font-family']) || $root_vars .= '--atlets-additional-font-family: ' . esc_attr($extra_font['font-family']) . ';';
        empty($extra_font['font-weight']) || $root_vars .= '--atlets-additional-font-weight: ' . esc_attr($extra_font['font-weight']) . ';';
        //* ↑ additional font variables

        /**
         * Footer Variables
         */
        if (
            Atlets::get_option('footer_switch')
            && 'widgets' === Atlets::get_option('footer_content_type')
        ) {
            $root_vars .= '--atlets-footer-content-color: ' . (esc_attr(Atlets::get_option('footer_text_color') ?? '')) . ';';
            $root_vars .= '--atlets-footer-heading-color: ' . (esc_attr(Atlets::get_option('footer_heading_color') ?? '')) . ';';
            $root_vars .= '--atlets-copyright-content-color: ' . (esc_attr(Atlets::get_mb_option('copyright_text_color', 'mb_copyright_switch', 'on') ?? '')) . ';';
        }
        //* ↑ footer variables

        /**
         * Encoded SVG variables
         */
        $root_vars .= '--atlets-svg-1: url(\'data:image/svg+xml; utf8, <svg version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px" viewBox="0 0 100 13.6" style="enable-background:new 0 0 100 13.6;" xml:space="preserve"><path class="st0" fill="' . str_replace('#', '%23', $theme_secondary_color) . '" d="M28.8,9.7L33,6.9l4.3,2.9c7.7,5.2,17.7,5.2,25.4,0L67,6.9l4.3,2.9c3.9,2.6,8.3,3.9,12.7,3.9c4.4,0,8.8-1.3,12.7-3.9l3.2-2.1C99.9,6.3,100,5,100,3.7c0-0.6,0-1.1,0-1.7l-0.6-0.9L93.5,5c-5.8,3.9-13.3,3.9-19.1,0L67,0l-7.4,5c-5.8,3.9-13.3,3.9-19.1,0L33,0l-7.4,5C19.8,8.9,12.3,8.9,6.5,5L0.7,1.1L0,2c0,0.6,0,1.1,0,1.7C0,5,0.1,6.3,0.2,7.6l3.2,2.1C11.1,14.9,21,14.9,28.8,9.7z"/></svg>\');';
        //* ↑ encoded SVG variables

        /**
         * Side Panel Variables
         */
        if (
            class_exists('RWMB_Loader')
            && get_queried_object_id() !== 0
            && 'custom' === rwmb_meta('mb_customize_side_panel')
        ) {
            $sidepanel_title_color = rwmb_meta('mb_side_panel_title_color');
        } else {
            $sidepanel_title_color = Atlets::get_option('side_panel_title_color')['rgba'] ?? '';
        }
        $root_vars .= '--atlets-sidepanel-title-color: ' . esc_attr($sidepanel_title_color) . ';';
        //* ↑ side panel variables

        /**
         * Elementor Container
         */
        $root_vars .= '--atlets-elementor-container-width: ' . $this->get_elementor_container_width() . 'px;';
        //* ↑ elementor container

        $css_variables = ':root {' . $root_vars . '}';

        $extra_css .= $this->get_mobile_header_extra_css();
        $extra_css .= $this->get_page_title_responsive_extra_css();

        return $css_variables . $extra_css;
    }

    public function get_elementor_container_width()
    {
        if (
            did_action('elementor/loaded')
            && defined('ELEMENTOR_VERSION')
        ) {
            if (version_compare(ELEMENTOR_VERSION, '3.0', '<')) {
                $container_width = get_option('elementor_container_width') ?: 1140;
            } else {
                $kit_id = (new \Elementor\Core\Kits\Manager())->get_active_id();
                $meta_key = \Elementor\Core\Settings\Page\Manager::META_KEY;
                $kit_settings = get_post_meta($kit_id, $meta_key, true);
                $container_width = $kit_settings['container_width']['size'] ?? 1140;
            }
        }

        return $container_width ?? 1170;
    }

    protected function get_mobile_header_extra_css()
    {
        $extra_css = '';

        if (Atlets::get_option('mobile_header')) {
            $mobile_background = Atlets::get_option('mobile_background')['rgba'] ?? '';
            $mobile_color = Atlets::get_option('mobile_color');

            $extra_css .= '.wgl-theme-header {'
                    . 'background-color: ' . esc_attr($mobile_background) . ' !important;'
                    . 'color: ' . esc_attr($mobile_color) . ' !important;'
                . '}';
        }

        $extra_css .= 'header.wgl-theme-header .wgl-mobile-header {'
                . 'display: block;'
            . '}'
            . '.wgl-site-header,'
            . '.wgl-theme-header .primary-nav {'
                . 'display: none;'
            . '}'
            . '.wgl-theme-header .hamburger-box {'
                . 'display: inline-block;'
            . '}'
            . 'header.wgl-theme-header .mobile_nav_wrapper .primary-nav {'
                . 'display: block;'
            . '}'
            . '.wgl-theme-header .wgl-sticky-header {'
                . 'display: none;'
            . '}'
            . '.wgl-page-socials {'
                . 'display: none;'
            . '}';

        $mobile_sticky = Atlets::get_option('mobile_sticky');

        if (Atlets::get_option('mobile_over_content')) {
            $extra_css .= 'body .wgl-theme-header {'
                    . 'position: absolute;'
                    . 'z-index: 99;'
                    . 'width: 100%;'
                    . 'left: 0;'
                    . 'top: 0;'
                . '}';

            if ($mobile_sticky) {
                $extra_css .= 'body .wgl-theme-header .wgl-mobile-header {'
                        . 'position: absolute;'
                        . 'left: 0;'
                        . 'width: 100%;'
                    . '}';
            }

        } else {
            $extra_css .= 'body .wgl-theme-header.header_overlap {'
                    . 'position: relative;'
                    . 'z-index: 2;'
                . '}';
        }

        if ($mobile_sticky) {
            $extra_css .= 'body .wgl-theme-header,'
                . 'body .wgl-theme-header.header_overlap {'
                    . 'position: sticky;'
                    . 'top: 0;'
                . '}';
        }

        return '@media only screen and (max-width: ' . $this->get_header_mobile_breakpoint() . 'px) {' . $extra_css . '}';
    }

    protected function get_header_mobile_breakpoint()
    {
        $elementor_breakpoint = '';

        if (
            'custom' === $this->header_type
            && $this->header_page_id
            && did_action('elementor/loaded')
        ) {
            $settings_manager = \Elementor\Core\Settings\Manager::get_settings_managers('page');
            $settings_model = $settings_manager->get_model($this->header_page_id);

            $elementor_breakpoint = $settings_model->get_settings('mobile_breakpoint');
        }

        return $elementor_breakpoint ?: (int) Atlets::get_option('header_mobile_queris');
    }

    protected function get_page_title_responsive_extra_css()
    {
        $page_title_resp = Atlets::get_option('page_title_resp_switch');

        if (
            $this->RWMB_is_active()
            && 'on' === rwmb_meta('mb_page_title_switch')
            && rwmb_meta('mb_page_title_resp_switch')
        ) {
            $page_title_resp = true;
        }

        if (!$page_title_resp) {
            // Bailout, if no any responsive logic
            return;
        }

        $pt_breakpoint = (int) Atlets::get_mb_option('page_title_resp_resolution', 'mb_page_title_resp_switch', true);
        $pt_padding = Atlets::get_mb_option('page_title_resp_padding', 'mb_page_title_resp_switch', true);
        $pt_margin_b = Atlets::get_mb_option('page_title_resp_margin', 'mb_page_title_resp_switch', true);
        $pt_font = Atlets::get_mb_option('page_title_resp_font', 'mb_page_title_resp_switch', true);

        $breadcrumbs_font = Atlets::get_mb_option('page_title_resp_breadcrumbs_font', 'mb_page_title_resp_switch', true);
        $breadcrumbs_switch = Atlets::get_mb_option('page_title_resp_breadcrumbs_switch', 'mb_page_title_resp_switch', true);

        //* Title styles
        $pt_color = !empty($pt_font['color']) ? 'color: ' . esc_attr($pt_font['color']) . ' !important;' : '';
        $pt_f_size = !empty($pt_font['font-size']) ? ' font-size: ' . esc_attr((int) $pt_font['font-size']) . 'px !important;' : '';
        $pt_line_height = !empty($pt_font['line-height']) ? ' line-height: ' . esc_attr((int) $pt_font['line-height']) . 'px !important;' : '';
        $pt_additional_style = !(bool) $breadcrumbs_switch ? ' margin-bottom: 0 !important;' : '';
        $title_style = $pt_color . $pt_f_size . $pt_line_height . $pt_additional_style;

        //* Breadcrumbs Styles
        $breadcrumbs_color = !empty($breadcrumbs_font['color']) ? 'color: ' . esc_attr($breadcrumbs_font['color']) . ' !important;' : '';
        $breadcrumbs_f_size = !empty($breadcrumbs_font['font-size']) ? 'font-size: ' . esc_attr((int) $breadcrumbs_font['font-size']) . 'px !important;' : '';
        $breadcrumbs_line_height = !empty($breadcrumbs_font['line-height']) ? 'line-height: ' . esc_attr((int) $breadcrumbs_font['line-height']) . 'px !important;' : '';
        $breadcrumbs_display = !(bool) $breadcrumbs_switch ? 'display: none !important;' : '';
        $breadcrumbs_style = $breadcrumbs_color . $breadcrumbs_f_size . $breadcrumbs_line_height . $breadcrumbs_display;

        //* Blog Single Type 3
        $blog_t3_padding_top = Atlets::get_option('single_padding_layout_3')['padding-top'] > 150 ? 150 : '';

        $extra_css = '.page-header {'
                . (!empty($pt_padding['padding-top']) ? 'padding-top: ' . esc_attr((int) $pt_padding['padding-top']) . 'px !important;' : '')
                . (!empty($pt_padding['padding-bottom']) ? 'padding-bottom: ' . esc_attr((int) $pt_padding['padding-bottom']) . 'px  !important;' : '')
                . (!empty($pt_margin_b['margin-bottom']) ? 'margin-bottom: ' . esc_attr((int) $pt_margin_b['margin-bottom']) . 'px  !important;' : '')
                . 'height: auto !important;'
            . '}'
            . '.page-header_content .page-header_title {'
                . $title_style
            . '}'
            . '.page-header_content .page-header_breadcrumbs {'
                . $breadcrumbs_style
            . '}'
            . '.page-header_breadcrumbs .divider:not(:last-child):before {'
                . 'width: 10px;'
            . '}';

        if ($blog_t3_padding_top) {
            $extra_css .= '.single-post .post_featured_bg > .blog-post {'
                    . 'padding-top: ' . $blog_t3_padding_top . 'px !important;'
                . '}';
        }

        return '@media (max-width: ' . $pt_breakpoint . 'px) {' . $extra_css . '}';
    }

    /**
     * Enqueue theme stylesheets
     *
     * Function keeps track of already enqueued stylesheets and stores them in `enqueued_stylesheets[]`
     *
     * @param string   $tag      Unprefixed handle.
     * @param string   $file_dir Path to stylesheet folder, relative to root folder.
     * @param string[] $deps     Optional. An array of registered stylesheet handles this stylesheet depends on.
     */
    public function enqueue_style($tag, $file_dir, $deps = [])
    {
        $prefixed_tag = $this->theme_slug . '-' . $tag;
        $this->enqueued_stylesheets[] = $prefixed_tag;

        wp_enqueue_style(
            $prefixed_tag,
            $this->template_directory_uri . $file_dir . $tag . $this->use_minified . '.css',
            $deps
        );
    }

    public function enqueue_pluggable_styles()
    {
        //* Preloader
        if (Atlets::get_option('preloader')) {
            $this->enqueue_style('preloader', '/css/pluggable/');
        }

        //* Page 404|Search
        if (is_404() || is_search()) {
            $this->enqueue_style('page-404', '/css/pluggable/');
        }

        //* Gutenberg
        if (Atlets::get_option('disable_wp_gutenberg')) {
            wp_dequeue_style('wp-block-library');
        } else {
            $this->enqueue_style('gutenberg', '/css/pluggable/');
        }

        //* Disable WooCommerce block styles (front-end)
        if (Atlets::get_option('disable_woocommerce_block')) {
            wp_dequeue_style( 'wc-block-style' );
        }

        //* Post Single (blog, portfolio)
        if (is_single()) {
            $post_type = get_post()->post_type;
            if ('post' === $post_type || 'portfolio' === $post_type) {
                $this->enqueue_style('blog-single-post', '/css/pluggable/');
            }
        }

        //* WooCommerce Plugin
        if (class_exists('WooCommerce')) {
            $this->enqueue_style('woocommerce', '/css/pluggable/');
        }

        //* Side Panel
        if (Atlets::get_option('side_panel_enable')) {
            $this->enqueue_style('side-panel', '/css/pluggable/');
        }

        //* WPML plugin
        if (class_exists('SitePress')) {
            $this->enqueue_style('wpml', '/css/pluggable/');
        }

        //* Polylang plugin
        if (function_exists('pll_the_languages')) {
            $this->enqueue_style('polylang', '/css/pluggable/');
        }
    }

    public function frontend_scripts()
    {
        wp_enqueue_script('atlets-theme-addons', $this->template_directory_uri . '/js/theme-addons' . $this->use_minified . '.js', ['jquery'], false, true);
        wp_enqueue_script('atlets-theme', $this->template_directory_uri . '/js/theme.js', ['jquery'], false, true);

        wp_localize_script('atlets-theme', 'wgl_core', [
            'ajaxurl' => esc_url(admin_url('admin-ajax.php')),
        ]);

        if (is_singular() && comments_open() && get_option('thread_comments')) {
            wp_enqueue_script('comment-reply');
        }

        wp_enqueue_script('perfect-scrollbar', $this->template_directory_uri . '/js/perfect-scrollbar.min.js');
    }

    public function admin_stylesheets()
    {
        wp_enqueue_style('atlets-admin', $this->template_directory_uri . '/core/admin/css/admin.css');
        wp_enqueue_style('font-awesome-5-all', $this->template_directory_uri . '/css/font-awesome-5.min.css');
        wp_enqueue_style('wp-color-picker');
    }

    public function admin_scripts()
    {
        wp_enqueue_media();

        wp_enqueue_script('wp-color-picker');
	    wp_localize_script('wp-color-picker', 'wpColorPickerL10n', [
		    'clear' => esc_html__('Clear', 'atlets'),
		    'clearAriaLabel' => esc_html__('Clear color', 'atlets'),
		    'defaultString' => esc_html__('Default', 'atlets'),
		    'defaultAriaLabel' => esc_html__('Select default color', 'atlets'),
		    'pick' => esc_html__('Select', 'atlets'),
		    'defaultLabel' => esc_html__('Color value', 'atlets'),
        ]);

        wp_enqueue_script('atlets-admin', $this->template_directory_uri . '/core/admin/js/admin.js');

        if (class_exists('RWMB_Loader')) {
            wp_enqueue_script('atlets-metaboxes', $this->template_directory_uri . '/core/admin/js/metaboxes.js');
        }

        $currentTheme = wp_get_theme();
        $theme_name = $currentTheme->parent() == false ? wp_get_theme()->get('Name') : wp_get_theme()->parent()->get('Name');
        $theme_name = trim($theme_name);

        $purchase_code = $email = '';
        if (Atlets::wgl_theme_activated()) {
            $theme_details = get_option('wgl_licence_validated');
            $purchase_code = $theme_details['purchase'] ?? '';
            $email = $theme_details['email'] ?? '';
        }

        wp_localize_script('atlets-admin', 'wgl_verify', [
            'ajaxurl' => esc_js(admin_url('admin-ajax.php')),
            'wglUrlActivate' => esc_js(Wgl_Theme_Verify::get_instance()->api . 'verification'),
            'wglUrlDeactivate' => esc_js(Wgl_Theme_Verify::get_instance()->api . 'deactivate'),
            'domainUrl' => esc_js(site_url('/')),
            'themeName' => esc_js($theme_name),
            'purchaseCode' => esc_js($purchase_code),
            'email' => esc_js($email),
            'message' => esc_js(esc_html__('Thank you, your license has been validated', 'atlets')),
            'ajax_nonce' => esc_js(wp_create_nonce('_notice_nonce'))
        ]);
    }

    protected function add_body_classes()
    {
        add_filter('body_class', function (Array $classes) {
            if ($this->gradient_enabled) {
                $classes[] = 'theme-gradient';
            }

            if (
                is_single()
                && 'post' === get_post_type(get_queried_object_id())
                && '3' === Atlets::get_mb_option('single_type_layout', 'mb_post_layout_conditional', 'custom')
            ) {
                $classes[] = 'atlets-blog-type-overlay';
            }

            return $classes;
        });
    }

    public function RWMB_is_active()
    {
        $id = !is_archive() ? get_queried_object_id() : 0;

        return class_exists('RWMB_Loader') && 0 !== $id;
    }

    public static function instance()
    {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }

        return self::$instance;
    }
}

new Atlets_Dynamic_Styles();
